<?php
session_start();

// Configuration
$subscriptionsFile = __DIR__ . '/data/whatsapp-subscriptions.json';
$passwordFile = __DIR__ . '/data/admin-password.txt';

// Fonction d'échappement HTML
function h($value): string {
    return htmlspecialchars((string)($value ?? ''), ENT_QUOTES, 'UTF-8');
}

// Fonction pour initialiser le mot de passe par défaut
function initPasswordFile($passwordFile) {
    if (!file_exists($passwordFile)) {
        // Mot de passe par défaut: "admin123" (à changer lors de la première connexion)
        $defaultHash = password_hash('admin123', PASSWORD_DEFAULT);
        $dir = dirname($passwordFile);
        if (!is_dir($dir)) {
            mkdir($dir, 0755, true);
        }
        file_put_contents($passwordFile, $defaultHash);
    }
}

// Initialiser le fichier de mot de passe si nécessaire
initPasswordFile($passwordFile);

// Vérifier l'authentification
$isAuthenticated = isset($_SESSION['admin_authenticated']) && $_SESSION['admin_authenticated'] === true;

// Gestion de la déconnexion
if (isset($_GET['logout'])) {
    session_destroy();
    header('Location: admin-whatsapp.php');
    exit;
}

// Gestion de la connexion
$loginError = null;
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['login'])) {
    $password = $_POST['password'] ?? '';
    $storedHash = file_get_contents($passwordFile);
    
    if (password_verify($password, $storedHash)) {
        $_SESSION['admin_authenticated'] = true;
        $_SESSION['admin_login_time'] = time();
        header('Location: admin-whatsapp.php');
        exit;
    } else {
        $loginError = 'Mot de passe incorrect';
    }
}

// Si pas authentifié, afficher la page de login
if (!$isAuthenticated) {
    ?>
<!doctype html>
<html lang="fr">
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <title>Connexion - Administration WhatsApp</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet" />
  <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.css" rel="stylesheet" />
  <style>
    :root {
      --brand-primary: #602596;
      --brand-secondary: #0093b4;
      --brand-accent: #fe5656;
      --brand-dark: #4e006f;
      --brand-light: #f8fafc;
      /* Compatibility */
      --brand-teal: #20B2AA;
      --brand-red: #fe5656;
      --brand-purple: #602596;
    }
     body {
       background: linear-gradient(135deg, rgba(96, 37, 150, 0.95), rgba(0, 147, 180, 0.85));
       min-height: 100vh;
       display: flex;
       align-items: center;
       justify-content: center;
       font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
     }
    .login-container {
      background: white;
      border-radius: 1rem;
      padding: 2.5rem;
      max-width: 400px;
      width: 100%;
      box-shadow: 0 20px 60px rgba(0,0,0,0.3);
    }
     .login-icon {
       width: 80px;
       height: 80px;
       background: linear-gradient(135deg, var(--brand-purple), var(--brand-secondary));
       border-radius: 50%;
       display: flex;
       align-items: center;
       justify-content: center;
       margin: 0 auto 1.5rem;
       color: white;
       font-size: 2rem;
     }
     .btn-login {
       background: linear-gradient(135deg, var(--brand-secondary), var(--brand-purple));
       color: white;
       border: none;
       width: 100%;
       padding: 0.75rem;
     }
     .btn-login:hover {
       transform: translateY(-2px);
       box-shadow: 0 4px 12px rgba(0, 147, 180, 0.3);
       color: white;
     }
    .alert {
      border-radius: 0.5rem;
    }
  </style>
</head>
<body>
  <div class="login-container">
    <div class="login-icon">
      <i class="bi bi-lock-fill"></i>
    </div>
     <h2 class="text-center mb-4" style="color: var(--brand-purple);">Connexion administrateur</h2>
    
    <?php if ($loginError): ?>
      <div class="alert alert-danger" role="alert">
        <i class="bi bi-exclamation-triangle me-2"></i><?= h($loginError) ?>
      </div>
    <?php endif; ?>
    
    <form method="post">
      <div class="mb-3">
        <label for="password" class="form-label">Mot de passe</label>
        <div class="input-group">
          <span class="input-group-text"><i class="bi bi-key"></i></span>
          <input 
            type="password" 
            class="form-control" 
            id="password" 
            name="password" 
            required 
            autofocus
            placeholder="Entrez votre mot de passe"
          />
        </div>
      </div>
      <button type="submit" name="login" class="btn btn-login">
        <i class="bi bi-box-arrow-in-right me-2"></i>Se connecter
      </button>
    </form>
    

  </div>
  <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
    <?php
    exit;
}

// Vérifier que la session n'est pas expirée (30 minutes)
if (isset($_SESSION['admin_login_time']) && (time() - $_SESSION['admin_login_time']) > 1800) {
    session_destroy();
    header('Location: admin-whatsapp.php');
    exit;
}

// Gestion du changement de mot de passe
$passwordChangeMessage = null;
$passwordChangeError = null;
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['change_password'])) {
    $currentPassword = $_POST['current_password'] ?? '';
    $newPassword = $_POST['new_password'] ?? '';
    $confirmPassword = $_POST['confirm_password'] ?? '';
    
    $storedHash = file_get_contents($passwordFile);
    
    if (!password_verify($currentPassword, $storedHash)) {
        $passwordChangeError = 'Le mot de passe actuel est incorrect';
    } elseif (empty($newPassword) || strlen($newPassword) < 6) {
        $passwordChangeError = 'Le nouveau mot de passe doit contenir au moins 6 caractères';
    } elseif ($newPassword !== $confirmPassword) {
        $passwordChangeError = 'Les nouveaux mots de passe ne correspondent pas';
    } else {
        $newHash = password_hash($newPassword, PASSWORD_DEFAULT);
        if (file_put_contents($passwordFile, $newHash, LOCK_EX) !== false) {
            $passwordChangeMessage = 'Mot de passe changé avec succès !';
        } else {
            $passwordChangeError = 'Erreur lors du changement de mot de passe';
        }
    }
}

// Gestion de l'export CSV (vérifier l'authentification)
if (isset($_GET['export']) && $_GET['export'] === 'csv') {
    if (!$isAuthenticated) {
        header('HTTP/1.1 403 Forbidden');
        die('Accès non autorisé');
    }
    
    if (!file_exists($subscriptionsFile)) {
        header('Content-Type: text/plain');
        die('Aucune donnée à exporter.');
    }
    
    $data = json_decode(file_get_contents($subscriptionsFile), true) ?: [];
    
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename="whatsapp-subscriptions-' . date('Y-m-d') . '.csv"');
    
    $output = fopen('php://output', 'w');
    
    // BOM UTF-8 pour Excel
    fprintf($output, chr(0xEF).chr(0xBB).chr(0xBF));
    
    // En-têtes
    fputcsv($output, ['Numéro WhatsApp', 'Date et Heure', 'Adresse IP'], ';');
    
    // Données
    foreach ($data as $row) {
        fputcsv($output, [
            $row['whatsapp'] ?? '',
            $row['timestamp'] ?? '',
            $row['ip'] ?? ''
        ], ';');
    }
    
    fclose($output);
    exit;
}

// Charger les données
$subscriptions = [];
if (file_exists($subscriptionsFile)) {
    $content = file_get_contents($subscriptionsFile);
    $subscriptions = json_decode($content, true) ?: [];
}

// Tri par date (plus récent en premier)
usort($subscriptions, function($a, $b) {
    $timeA = strtotime($a['timestamp'] ?? '');
    $timeB = strtotime($b['timestamp'] ?? '');
    return $timeB - $timeA;
});

// Recherche et filtrage
$search = trim($_GET['search'] ?? '');
$filteredSubscriptions = $subscriptions;

if (!empty($search)) {
    $searchLower = mb_strtolower($search);
    $filteredSubscriptions = array_filter($subscriptions, function($sub) use ($searchLower) {
        return (
            strpos(mb_strtolower($sub['whatsapp'] ?? ''), $searchLower) !== false ||
            strpos(mb_strtolower($sub['timestamp'] ?? ''), $searchLower) !== false ||
            strpos(mb_strtolower($sub['ip'] ?? ''), $searchLower) !== false
        );
    });
}

// Statistiques
$totalCount = count($subscriptions);
$filteredCount = count($filteredSubscriptions);
$todayCount = 0;
$weekCount = 0;
$uniqueNumbers = [];

foreach ($subscriptions as $sub) {
    $timestamp = strtotime($sub['timestamp'] ?? '');
    $now = time();
    
    // Aujourd'hui
    if ($timestamp >= strtotime('today')) {
        $todayCount++;
    }
    
    // Cette semaine
    if ($timestamp >= strtotime('monday this week')) {
        $weekCount++;
    }
    
    // Numéros uniques
    $cleanNumber = preg_replace('/[^0-9]/', '', $sub['whatsapp'] ?? '');
    if (!empty($cleanNumber)) {
        $uniqueNumbers[$cleanNumber] = true;
    }
}

$uniqueCount = count($uniqueNumbers);
?>
<!doctype html>
<html lang="fr">
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <title>Administration - Inscriptions WhatsApp</title>
  
  <!-- Bootstrap 5 & Icons -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet" />
  <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.css" rel="stylesheet" />
  
  <style>
    :root {
      --brand-dark: #0b3d2e;
      --brand-highlight: #2ebd8a;
    }
    
    body {
      background: #f4f6f8;
      font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
    }
    
     .bg-brand {
       background: var(--brand-purple);
     }
    
     .stat-card {
       background: white;
       border-radius: 0.75rem;
       padding: 1.5rem;
       box-shadow: 0 2px 8px rgba(0,0,0,0.1);
       border-left: 4px solid var(--brand-secondary);
     }
     
     .stat-number {
       font-size: 2rem;
       font-weight: 800;
       color: var(--brand-secondary);
     }
    
    .stat-label {
      color: #6c757d;
      font-size: 0.9rem;
      margin-top: 0.5rem;
    }
    
    .table-container {
      background: white;
      border-radius: 0.75rem;
      box-shadow: 0 2px 8px rgba(0,0,0,0.1);
      overflow: hidden;
    }
    
     .table thead {
       background: var(--brand-purple);
       color: white;
     }
    
    .table thead th {
      border: none;
      font-weight: 600;
      padding: 1rem;
    }
    
    .table tbody td {
      padding: 1rem;
      vertical-align: middle;
    }
    
    .table tbody tr:hover {
      background: #f8f9fa;
    }
    
    .badge-whatsapp {
      background: #25D366;
      color: white;
    }
    
    .search-box {
      position: relative;
    }
    
    .search-box .form-control {
      padding-left: 2.5rem;
    }
    
    .search-box i {
      position: absolute;
      left: 0.75rem;
      top: 50%;
      transform: translateY(-50%);
      color: #6c757d;
    }
    
     .btn-brand {
       background: linear-gradient(135deg, var(--brand-secondary), var(--brand-purple));
       color: white;
       border: none;
     }
     
     .btn-brand:hover {
       background: linear-gradient(135deg, var(--brand-purple), var(--brand-secondary));
       color: white;
       transform: translateY(-1px);
       box-shadow: 0 4px 12px rgba(0, 147, 180, 0.3);
     }
    
    .btn-export {
      background: #28a745;
      color: white;
      border: none;
    }
    
    .btn-export:hover {
      background: #218838;
      color: white;
    }
  </style>
</head>
<body>
  <div class="container-fluid py-4">
    <div class="row mb-4">
      <div class="col-12">
        <div class="d-flex justify-content-between align-items-center mb-3">
          <div>
            <h1 class="h3 mb-1">
              <i class="bi bi-whatsapp text-success"></i>
              Inscriptions WhatsApp
            </h1>
            <p class="text-muted mb-0">Gestion et consultation des abonnés</p>
          </div>
                     <div>
             <a href="coming-soon.php" class="btn btn-outline-secondary">
               <i class="bi bi-arrow-left"></i> Retour
             </a>
             <a href="?export=csv" class="btn btn-export">
               <i class="bi bi-download"></i> Exporter CSV
             </a>
             <button type="button" class="btn btn-outline-primary" data-bs-toggle="modal" data-bs-target="#changePasswordModal">
               <i class="bi bi-key"></i> Changer le mot de passe
             </button>
             <a href="?logout=1" class="btn btn-outline-danger" onclick="return confirm('Êtes-vous sûr de vouloir vous déconnecter ?')">
               <i class="bi bi-box-arrow-right"></i> Déconnexion
             </a>
           </div>
        </div>
      </div>
    </div>

    <!-- Statistiques -->
    <div class="row g-3 mb-4">
      <div class="col-md-3 col-sm-6">
        <div class="stat-card">
          <div class="stat-number"><?= number_format($totalCount) ?></div>
          <div class="stat-label">
            <i class="bi bi-people-fill"></i> Total inscriptions
          </div>
        </div>
      </div>
      <div class="col-md-3 col-sm-6">
        <div class="stat-card">
          <div class="stat-number"><?= number_format($uniqueCount) ?></div>
          <div class="stat-label">
            <i class="bi bi-person-check-fill"></i> Numéros uniques
          </div>
        </div>
      </div>
      <div class="col-md-3 col-sm-6">
        <div class="stat-card">
          <div class="stat-number"><?= number_format($todayCount) ?></div>
          <div class="stat-label">
            <i class="bi bi-calendar-day"></i> Aujourd'hui
          </div>
        </div>
      </div>
      <div class="col-md-3 col-sm-6">
        <div class="stat-card">
          <div class="stat-number"><?= number_format($weekCount) ?></div>
          <div class="stat-label">
            <i class="bi bi-calendar-week"></i> Cette semaine
          </div>
        </div>
      </div>
    </div>

    <!-- Messages de changement de mot de passe -->
    <?php if ($passwordChangeMessage || $passwordChangeError): ?>
      <div class="row mb-3">
        <div class="col-12">
          <?php if ($passwordChangeMessage): ?>
            <div class="alert alert-success alert-dismissible fade show" role="alert">
              <i class="bi bi-check-circle me-2"></i><?= h($passwordChangeMessage) ?>
              <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
          <?php endif; ?>
          <?php if ($passwordChangeError): ?>
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
              <i class="bi bi-exclamation-triangle me-2"></i><?= h($passwordChangeError) ?>
              <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
          <?php endif; ?>
        </div>
      </div>
    <?php endif; ?>

    <!-- Recherche -->
    <div class="row mb-3">
      <div class="col-12">
        <div class="table-container p-3">
          <form method="get" class="d-flex gap-2">
            <div class="search-box flex-grow-1">
              <i class="bi bi-search"></i>
              <input 
                type="text" 
                name="search" 
                class="form-control" 
                placeholder="Rechercher par numéro, date ou IP..." 
                value="<?= h($search) ?>"
              />
            </div>
            <button type="submit" class="btn btn-brand">
              <i class="bi bi-search"></i> Rechercher
            </button>
            <?php if (!empty($search)): ?>
              <a href="?" class="btn btn-outline-secondary">
                <i class="bi bi-x-lg"></i> Effacer
              </a>
            <?php endif; ?>
          </form>
          <?php if (!empty($search)): ?>
            <div class="mt-2">
              <small class="text-muted">
                <i class="bi bi-funnel"></i> 
                <?= number_format($filteredCount) ?> résultat(s) sur <?= number_format($totalCount) ?>
              </small>
            </div>
          <?php endif; ?>
        </div>
      </div>
    </div>

    <!-- Tableau -->
    <div class="row">
      <div class="col-12">
        <div class="table-container">
          <?php if (empty($filteredSubscriptions)): ?>
            <div class="p-5 text-center text-muted">
              <i class="bi bi-inbox" style="font-size: 3rem;"></i>
              <p class="mt-3 mb-0">
                <?php if (!empty($search)): ?>
                  Aucun résultat trouvé pour "<?= h($search) ?>"
                <?php else: ?>
                  Aucune inscription pour le moment
                <?php endif; ?>
              </p>
            </div>
          <?php else: ?>
            <div class="table-responsive">
              <table class="table table-hover mb-0">
                <thead>
                  <tr>
                    <th>#</th>
                    <th>
                      <i class="bi bi-whatsapp"></i> Numéro WhatsApp
                    </th>
                    <th>
                      <i class="bi bi-calendar3"></i> Date et Heure
                    </th>
                    <th>
                      <i class="bi bi-globe"></i> Adresse IP
                    </th>
                    <th>
                      <i class="bi bi-link-45deg"></i> Actions
                    </th>
                  </tr>
                </thead>
                <tbody>
                  <?php 
                  $index = 0;
                  foreach ($filteredSubscriptions as $subscription): 
                    $index++;
                    $whatsapp = $subscription['whatsapp'] ?? '';
                    $cleanNumber = preg_replace('/[^0-9]/', '', $whatsapp);
                    $whatsappLink = 'https://wa.me/' . $cleanNumber;
                    $timestamp = $subscription['timestamp'] ?? '';
                    $dateObj = new DateTime($timestamp);
                  ?>
                    <tr>
                      <td class="text-muted"><?= number_format($index) ?></td>
                      <td>
                        <span class="badge badge-whatsapp">
                          <i class="bi bi-whatsapp"></i> <?= h($whatsapp) ?>
                        </span>
                      </td>
                      <td>
                        <div>
                          <strong><?= $dateObj->format('d/m/Y') ?></strong>
                          <small class="text-muted d-block"><?= $dateObj->format('H:i:s') ?></small>
                        </div>
                      </td>
                      <td>
                        <code class="text-muted"><?= h($subscription['ip'] ?? '') ?></code>
                      </td>
                      <td>
                        <a 
                          href="<?= $whatsappLink ?>" 
                          target="_blank" 
                          rel="noopener"
                          class="btn btn-sm btn-success"
                          title="Ouvrir dans WhatsApp"
                        >
                          <i class="bi bi-whatsapp"></i> Contacter
                        </a>
                      </td>
                    </tr>
                  <?php endforeach; ?>
                </tbody>
              </table>
            </div>
          <?php endif; ?>
        </div>
      </div>
    </div>

    <!-- Pagination info -->
    <?php if (!empty($filteredSubscriptions) && $filteredCount > 10): ?>
      <div class="row mt-3">
        <div class="col-12 text-center text-muted">
          <small>
            Affichage de toutes les inscriptions (<?= number_format($filteredCount) ?> résultat<?= $filteredCount > 1 ? 's' : '' ?>)
          </small>
        </div>
      </div>
    <?php endif; ?>
  </div>

  <!-- Modal de changement de mot de passe -->
  <div class="modal fade" id="changePasswordModal" tabindex="-1" aria-labelledby="changePasswordModalLabel" aria-hidden="true">
    <div class="modal-dialog">
      <div class="modal-content">
        <div class="modal-header" style="background: var(--brand-dark); color: white;">
          <h5 class="modal-title" id="changePasswordModalLabel">
            <i class="bi bi-key"></i> Changer le mot de passe
          </h5>
          <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Fermer"></button>
        </div>
        <form method="post">
          <div class="modal-body">
            <?php if ($passwordChangeMessage): ?>
              <div class="alert alert-success alert-dismissible fade show" role="alert">
                <i class="bi bi-check-circle me-2"></i><?= h($passwordChangeMessage) ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
              </div>
            <?php endif; ?>
            <?php if ($passwordChangeError): ?>
              <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <i class="bi bi-exclamation-triangle me-2"></i><?= h($passwordChangeError) ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
              </div>
            <?php endif; ?>
            <div class="mb-3">
              <label for="current_password" class="form-label">Mot de passe actuel</label>
              <input type="password" class="form-control" id="current_password" name="current_password" required autofocus>
            </div>
            <div class="mb-3">
              <label for="new_password" class="form-label">Nouveau mot de passe</label>
              <input type="password" class="form-control" id="new_password" name="new_password" required minlength="6">
              <small class="text-muted">Minimum 6 caractères</small>
            </div>
            <div class="mb-3">
              <label for="confirm_password" class="form-label">Confirmer le nouveau mot de passe</label>
              <input type="password" class="form-control" id="confirm_password" name="confirm_password" required minlength="6">
            </div>
          </div>
          <div class="modal-footer">
            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Annuler</button>
            <button type="submit" name="change_password" class="btn btn-brand">
              <i class="bi bi-save me-2"></i>Changer le mot de passe
            </button>
          </div>
        </form>
      </div>
    </div>
  </div>

  <!-- Bootstrap JS -->
  <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
  
  <script>
    // Confirmation avant export
    document.querySelectorAll('a[href*="export=csv"]').forEach(link => {
      link.addEventListener('click', function(e) {
        if (!confirm('Voulez-vous exporter toutes les inscriptions au format CSV ?')) {
          e.preventDefault();
        }
      });
    });
    
    // Réinitialiser le modal quand il est fermé
    const passwordModal = document.getElementById('changePasswordModal');
    if (passwordModal) {
      passwordModal.addEventListener('hidden.bs.modal', function () {
        const form = this.querySelector('form');
        if (form) {
          form.reset();
          // Supprimer les messages d'alerte
          const alerts = this.querySelectorAll('.alert');
          alerts.forEach(alert => alert.remove());
        }
      });
      
      // Si un message de succès est affiché, ouvrir le modal
      <?php if ($passwordChangeMessage || $passwordChangeError): ?>
      const modal = new bootstrap.Modal(passwordModal);
      modal.show();
      <?php endif; ?>
    }
  </script>
</body>
</html>
